<?php
namespace FreePBX\modules\Sms;
//	License for all code of this FreePBX module can be found in the license file inside the module directory
//	Copyright 2013 Schmooze Com Inc.
//
abstract class AdaptorBase {
	public function __construct() {
		$this->db = \FreePBX::Database();
		if(!class_exists('Emojione\Emojione')) {
			include __DIR__."/Emojione.class.php";
		}
		$this->emoji = new \Emojione\Client(new \Emojione\Ruleset());
	}

	/**
	 * Insert message media into the database
	 * @param  integer $to    The DID the message was sent to
	 * @param  integer $from  The DID the message was from
	 * @param  string $cnam  The CNAME of the message
	 * @param  string $message The message body
	 * @param  array  $files Array of file names to process
	 * @param  integer $time    Unix Timestamp when the message was set (Use null for NOW())
	 * @param  string $adaptor The adaptor used to send the message
	 * @param  string $emid    External message id if there is one
	 * @return integer        The inserted message ID
	 */
	public function sendMedia($to,$from,$cnam,$message=null,$files=array(),$time=null,$adaptor=null,$emid=null) {
		$id = self::sendMessage($to,$from,$cnam,$message,$time,$adaptor,$emid);
		foreach($files as $file){
			if(file_exists($file)) {
				$data = file_get_contents($file);
				try {
					$sql = "INSERT INTO sms_media (`mid`, `name`, `raw`) VALUES (?, ?, ?)";
					$sth = $this->db->prepare($sql);
					$sth->execute(array($id, basename($file), $data));
				} catch (\Exception $e) {
					throw new \Exception('Unable to Insert Message Media into DB');
				}
				unlink($file);
			}
		}
		return $id;
	}

	/**
	 * Insert a sent message into the database
	 * @param  integer $to      The DID the message was sent to
	 * @param  integer $from    The DID the message was from
	 * @param  string $cnam    The CNAME of the message
	 * @param  string $message The message
	 * @param  integer $time    Unix Timestamp when the message was set (Use null for NOW())
	 * @param  string $adaptor The adaptor used to send the message
	 * @param  string $emid    External message id if there is one
	 * @return integer          The ID of the row that was inserted
	 */
	public function sendMessage($to,$from,$cnam,$message,$time=null,$adaptor=null,$emid=null) {
		$sql = "SELECT id FROM sms_dids WHERE `did` = :did";
		$sth = $this->db->prepare($sql);
		$sth->execute(array(
			":did" => $from
		));
		$didid = $sth->fetchColumn();

		$threadid = $this->determineThreadID($to,$from,'out');
		$sql = "INSERT INTO sms_messages (`from`, `to`, `cnam`, `direction`, `tx_rx_datetime`, `body`, `adaptor`, `emid`, `threadid`, `didid`,`timestamp`) VALUES (?, ?, ?, 'out', from_unixtime(?), ?, ?, ?, ?, ?, ?)";
		try {
			$sth = $this->db->prepare($sql);
			$message = $this->emoji->toShort($message);
			$time = !empty($time) ? $time : time();
			$message = !is_null($message) ? $message : "";
			$sth->execute(array($from, $to, $cnam, $time, $message, $adaptor, $emid, $threadid, $didid, $time));
			$id = $this->db->lastInsertId();
			\FreePBX::Hooks()->processHooks($id,$to,$from,$cnam,$message,$time,$adaptor,$emid,$threadid,$didid);
			\FreePBX::create()->astman->UserEvent("sms-outbound",array(
				"id" => $id,
				"to" => $to,
				"from" => $from,
				"cnam" => $cnam,
				"message" => $message,
				"time" => $time,
				"adaptor" => $adaptor,
				"emid" => !is_null($emid) ? $emid : 'sms-'.uniqid(),
				"threadid" => $threadid,
				"didid" => $didid
			));
			return $id;
		} catch (\Exception $e) {
			throw new \Exception('Unable to Insert Message into DB '.$e->getMessage());
		}
	}

	/**
	 * Insert a received message into the database
	 * @param  integer $to      The DID the message was sent to
	 * @param  integer $from    The DID the message was from
	 * @param  string $cnam    The CNAME of the message
	 * @param  string $message The message
	 * @param  integer $time    Unix Timestamp when the message was set (Use null for NOW())
	 * @param  string $adaptor The adaptor used to send the message
	 * @param  string $emid    External message id if there is one
	 * @return integer          The ID of the row that was inserted
	 */
	public function getMessage($to,$from,$cnam,$message,$time=null,$adaptor=null,$emid=null) {
		$sql = "SELECT id FROM sms_dids WHERE `did` = :did";
		$sth = $this->db->prepare($sql);
		$sth->execute(array(
			":did" => $to
		));
		$didid = $sth->fetchColumn();

		$threadid = $this->determineThreadID($to,$from,'in');
		$sql = "INSERT INTO sms_messages (`from`, `to`, `cnam`, `direction`, `tx_rx_datetime`, `body`, `adaptor`, `emid`, `threadid`, `didid`, `timestamp`) VALUES (?, ?, ?, 'in', from_unixtime(?), ?, ?, ?, ?, ?, ?)";
		try {
			$sth = $this->db->prepare($sql);
			$message = $this->emoji->toShort($message);
			$time = !empty($time) ? $time : time();
			$sth->execute(array($from, $to, $cnam, $time, $message, $adaptor, $emid, $threadid, $didid, $time));
			$id = $this->db->lastInsertId();
			\FreePBX::Hooks()->processHooks($id,$to,$from,$cnam,$message,$time,$adaptor,$emid,$threadid, $didid);
			\FreePBX::create()->astman->UserEvent("sms-inbound",array(
				"id" => $id,
				"to" => $to,
				"from" => $from,
				"cnam" => $cnam,
				"message" => $message,
				"time" => $time,
				"adaptor" => $adaptor,
				"emid" => !is_null($emid) ? $emid : 'sms-'.uniqid(),
				"threadid" => $threadid,
				"didid" => $didid
			));
			return $id;
		} catch (\Exception $e) {
			throw new \Exception('Unable to Insert Message into DB'.$e->getMessage());
		}
	}

	public function updateMessageByEMID($emid,$message,$adaptor=null) {
		$sql = "SELECT id FROM sms_messages WHERE `adaptor` = :adaptor AND `emid` = :emid";
		$sth = $this->db->prepare($sql);
		$sth->execute(array(":adaptor" => $adaptor, ":emid" => $emid));
		$res = $sth->fetch(\PDO::FETCH_ASSOC);
		if(empty($res)) {
			throw new \Exception("Invalid EMID");
		}

		$sql = "UPDATE sms_messages SET `body` = :body WHERE `emid` = :emid AND `adaptor` = :adaptor";
		$sth = $this->db->prepare($sql);
		$sth->execute(array(":body" => $message, ":adaptor" => $adaptor, ":emid" => $emid));
		return $res['id'];
	}

	/**
	 * Add Media into the database for sent or received messages
	 * @param integer $msgid The message ID from getMessage or sendMessage
	 * @param string $name  The filename
	 * @param string $data  The raw data from the file
	 */
	public function addMedia($msgid, $name, $data) {
		$sql = "INSERT INTO sms_media (`mid`, `name`, `raw`) VALUES (?, ?, ?)";
		try {
			$sth = $this->db->prepare($sql);
			$sth->execute(array($msgid, $name, $data));
		} catch (\Exception $e) {
			throw new \Exception('Unable to Insert Media into DB');
		}
	}

	/**
	 * Hooks for adaptor classes to modify dialplan
	 */
	public function dialPlanHooks(&$ext, $engine, $priority) {}

	/**
	 * Determine the thread ID
	 * @method determineThreadID
	 * @param  string            $to        Who the message was to
	 * @param  string            $from      Who the message was from
	 * @param  string            $direction The direction of the message
	 * @return string                       The resulting threadID
	 */
	private function determineThreadID($to,$from,$direction) {
		if($direction == 'in') {
			$local = $to;
			$remote = $from;
		} else {
			$local = $from;
			$remote = $to;
		}
		$threadid = sha1($local.$remote);
		return $threadid;
	}
}
