<?php
// vim: set ai ts=4 sw=4 ft=php:
//	License for all code of this FreePBX module can be found in the license file inside the module directory
//	Copyright 2014 Schmooze Com Inc.
//
namespace FreePBX\modules;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;
//progress bar
use Symfony\Component\Console\Helper\ProgressBar;
class Ucpnode extends \FreePBX_Helpers implements \BMO {
	private $nodever = "0.12.18";
	private $npmver = "2.15.11";
	private $icuver = "4.2.1";
	private $foreverroot = "/tmp";
	private $nodeloc = "/tmp";
	public function __construct($freepbx = null) {
		$this->db = $freepbx->Database;
		$this->freepbx = $freepbx;
		$this->foreverroot = $this->freepbx->Config->get('ASTVARLIBDIR') . "/ucp";
		$this->nodeloc = __DIR__."/node";
	}

	public function install() {
		global $amp_conf;
		if(file_exists($amp_conf['AMPBIN']."/freepbx_engine_hook_ucpnode") && is_writable($amp_conf['AMPBIN']."/freepbx_engine_hook_ucpnode")) {
			unlink($amp_conf['AMPBIN']."/freepbx_engine_hook_ucpnode");
		}

		$output = exec("node --version"); //v0.10.29
		$output = str_replace("v","",trim($output));
		if(empty($output)) {
			out(_("Node is not installed"));
			return false;
		}
		if(version_compare($output,$this->nodever,"<")) {
			out(sprintf(_("Node version is: %s requirement is %s. Run 'yum upgrade nodejs' from the CLI as root"),$output,$this->nodever));
			return false;
		}


		$output = exec("npm --version"); //v0.10.29
		$output = trim($output);
		if(empty($output)) {
			out(_("Node Package Manager is not installed"));
			return false;
		}
		if(version_compare($output,$this->npmver,"<")) {
			out(sprintf(_("NPM version is: %s requirement is %s. Run 'yum upgrade nodejs' from the CLI as root"),$output,$this->npmver));
			return false;
		}

		$output = exec("icu-config --version"); //v4.2.1
		$output = trim($output);
		if(empty($output)) {
			out(_("icu is not installed. You need to run: yum install icu libicu-devel"));
			return false;
		}
		if(version_compare($output,$this->icuver,"<")) {
			out(sprintf(_("ICU version is: %s requirement is %s"),$output,$this->icuver));
			return false;
		}

		$webgroup = $this->freepbx->Config->get('AMPASTERISKWEBGROUP');

		$data = posix_getgrgid(filegroup($this->getHomeDir()));
		if($data['name'] != $webgroup) {
			out(sprintf(_("Home directory [%s] is not writable"),$this->getHomeDir()));
			return false;
		}

		if(file_exists($this->getHomeDir()."/.npm")) {
			$data = posix_getgrgid(filegroup($this->getHomeDir()."/.npm"));
			if($data['name'] != $webgroup) {
				out(sprintf(_("Home directory [%s] is not writable"),$this->getHomeDir()."/.npm"));
				return false;
			}
		}

		outn(_("Installing/Updating Required Libraries. This may take a while..."));
		if (php_sapi_name() == "cli") {
			out("The following messages are ONLY FOR DEBUGGING. Ignore anything that says 'WARN' or is just a warning");
		}

		$command = $this->generateRunAsAsteriskCommand('npm-cache -v');
		$process = new Process($command);
		try {
			$process->mustRun();
		} catch (ProcessFailedException $e) {
			$command = $this->generateRunAsAsteriskCommand('npm install -g npm-cache 2>&1');
			exec($command);
		}

		$command = $this->generateRunAsAsteriskCommand('npm-cache -v');
		$process = new Process($command);
		try {
			$process->mustRun();
		} catch (ProcessFailedException $e) {
			out($e->getMessage());
			return false;
		}

		file_put_contents($this->nodeloc."/logs/install.log","");

		$command = $this->generateRunAsAsteriskCommand('npm-cache install 2>&1');
		$handle = popen($command, "r");
		$log = fopen($this->nodeloc."/logs/install.log", "a");
		while (($buffer = fgets($handle, 4096)) !== false) {
			fwrite($log,$buffer);
			if (php_sapi_name() == "cli") {
				outn($buffer);
			} else {
				outn(".");
			}
		}
		fclose($log);
		out("");
		out(_("Finished updating libraries!"));
		if(!file_exists($this->nodeloc."/node_modules/forever/bin/forever")) {
			out("");
			out(sprintf(_("There was an error installing. Please review the install log. (%s)"),$this->nodeloc."/logs/install.log"));
			return false;
		}

		if(!is_executable($this->nodeloc."/node_modules/forever/bin/forever")) {
			chmod($this->nodeloc."/node_modules/forever/bin/forever",0755);
		}
		if(!is_executable($this->nodeloc."/node_modules/forever/bin/monitor")) {
			chmod($this->nodeloc."/node_modules/forever/bin/monitor",0755);
		}

		$set = array();
		$set['module'] = 'ucpnode';
		$set['category'] = 'UCP NodeJS Server';

		// NODEJSENABLED
		$set['value'] = true;
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'Enable the NodeJS Server';
		$set['description'] = 'Whether to enable the backend server for UCP which allows instantaneous updates to the interface';
		$set['emptyok'] = 0;
		$set['level'] = 1;
		$set['readonly'] = 0;
		$set['type'] = CONF_TYPE_BOOL;
		$this->freepbx->Config->define_conf_setting('NODEJSENABLED',$set);

		// NODEJSTLSENABLED
		$set['value'] = false;
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'Enable TLS for the NodeJS Server';
		$set['description'] = 'Whether to enable TLS for the backend server for UCP which allows instantaneous updates to the interface';
		$set['emptyok'] = 0;
		$set['level'] = 1;
		$set['readonly'] = 0;
		$set['type'] = CONF_TYPE_BOOL;
		$this->freepbx->Config->define_conf_setting('NODEJSTLSENABLED',$set);

		// NODEJSBINDADDRESS
		$set['value'] = '::';
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'NodeJS Bind Address';
		$set['description'] = 'Address to bind to. Default is "::" (Listen for all IPv4 and IPv6 Connections)';
		$set['emptyok'] = 0;
		$set['type'] = CONF_TYPE_TEXT;
		$set['level'] = 2;
		$set['readonly'] = 0;
		$this->freepbx->Config->define_conf_setting('NODEJSBINDADDRESS',$set);

		// NODEJSBINDPORT
		$set['value'] = '8001';
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'NodeJS Bind Port';
		$set['description'] = 'Port to bind to. Default is 8001';
		$set['emptyok'] = 0;
		$set['options'] = array(10,65536);
		$set['type'] = CONF_TYPE_INT;
		$set['level'] = 2;
		$set['readonly'] = 0;
		$this->freepbx->Config->define_conf_setting('NODEJSBINDPORT',$set);

		// NODEJSHTTPSBINDADDRESS
		$set['value'] = '::';
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'NodeJS HTTPS Bind Address';
		$set['description'] = 'Address to bind to. Default is "::" (Listen for all IPv4 and IPv6 Connections)';
		$set['emptyok'] = 0;
		$set['type'] = CONF_TYPE_TEXT;
		$set['level'] = 2;
		$set['readonly'] = 0;
		$this->freepbx->Config->define_conf_setting('NODEJSHTTPSBINDADDRESS',$set);

		// NODEJSHTTPSBINDPORT
		$set['value'] = '8003';
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'NodeJS HTTPS Bind Port';
		$set['description'] = 'Port to bind to. Default is 8003';
		$set['emptyok'] = 0;
		$set['options'] = array(10,65536);
		$set['type'] = CONF_TYPE_INT;
		$set['level'] = 2;
		$set['readonly'] = 0;
		$this->freepbx->Config->define_conf_setting('NODEJSHTTPSBINDPORT',$set);

		// NODEJSTLSCERTFILE
		$set['value'] = '';
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'NodeJS HTTPS TLS Certificate Location';
		$set['description'] = 'Sets the path to the HTTPS server certificate. This is required if "Enable TLS for the NodeJS Server" is set to yes.';
		$set['emptyok'] = 1;
		$set['type'] = CONF_TYPE_TEXT;
		$set['level'] = 2;
		$set['readonly'] = 0;
		$this->freepbx->Config->define_conf_setting('NODEJSTLSCERTFILE',$set);

		// NODEJSTLSPRIVATEKEY
		$set['value'] = '';
		$set['defaultval'] =& $set['value'];
		$set['options'] = '';
		$set['name'] = 'NodeJS HTTPS TLS Private Key Location';
		$set['description'] = 'Sets the path to the HTTPS private key. This is required if "Enable TLS for the NodeJS Server" is set to yes.';
		$set['emptyok'] = 1;
		$set['type'] = CONF_TYPE_TEXT;
		$set['level'] = 2;
		$set['readonly'] = 0;
		$this->freepbx->Config->define_conf_setting('NODEJSTLSPRIVATEKEY',$set);

		$this->freepbx->Config->commit_conf_settings();

		$cert = $this->freepbx->Certman->getDefaultCertDetails();
		if(!empty($cert)) {
			$this->setDefaultCert($cert);
		}

		//need forever to be executable. it's sooo cutable
		if(!is_executable($this->nodeloc."/node_modules/forever/bin/forever")) {
			chmod($this->nodeloc."/node_modules/forever/bin/forever",0755);
		}

		if($this->freepbx->Modules->checkStatus("sysadmin")) {
			touch("/var/spool/asterisk/incron/ucpnode.logrotate");
		}

		//If we are root then start it as asterisk, otherwise we arent root so start it as the web user (all we can do really)
		outn(_("Stopping old running processes..."));
		$this->stopFreepbx();
		out(_("Done"));

		outn(_("Starting new UCP Node Process..."));
		$started = $this->startFreepbx();
		if(!$started) {
			out(_("Failed!"));
		} else {
			out(_("Started!"));
		}
	}
	public function uninstall() {
		outn(_("Stopping old running processes..."));
		$this->stopFreepbx();
		out(_("Done"));
		exec("rm -Rf ".$this->nodeloc."/node_modules");
	}
	public function backup(){

	}
	public function restore($backup){

	}

	public function setDefaultCert($details) {
		$certF = $details['integration']['files']['crt'];
		$keyF = $details['integration']['files']['key'];
		$this->freepbx->Config->update("NODEJSTLSENABLED",true);
		$this->freepbx->Config->update("NODEJSTLSCERTFILE",$certF);
		$this->freepbx->Config->update("NODEJSTLSPRIVATEKEY",$keyF);
	}

	public function getHomeDir() {
		$webuser = \FreePBX::Freepbx_conf()->get('AMPASTERISKWEBUSER');
		$web = posix_getpwnam($webuser);
		$home = trim($web['dir']);
		if (!is_dir($home)) {
			// Well, that's handy. It doesn't exist. Let's use ASTSPOOLDIR instead, because
			// that should exist and be writable.
			$home = \FreePBX::Freepbx_conf()->get('ASTSPOOLDIR');
			if (!is_dir($home)) {
				// OK, I give up.
				throw new \Exception(sprintf(_("Asterisk home dir (%s) doesn't exist, and, ASTSPOOLDIR doesn't exist. Aborting"),$home));
			}
		}
		return $home;
	}

	public function dashboardService() {
		if(!file_exists($this->nodeloc."/node_modules/forever")) {
			$service = array_merge($service, $this->genAlertGlyphicon('critical', _("UCP Node is damaged. Please reinstall.")));
			return array($service);
		}

		$service = array(
			'title' => _('UCP Daemon'),
			'type' => 'unknown',
			'tooltip' => _("Unknown"),
			'order' => 999,
			'glyph-class' => ''
		);
		$data = $this->getServiceStatus();
		if(!empty($data)) {
			$uptime = $this->get_date_diff(time(), (int)round($data['ctime']/1000));
			$service = array_merge($service, $this->genAlertGlyphicon('ok', sprintf(_("Running (Uptime: %s)"),$uptime)));
		} else {
			$service = array_merge($service, $this->genAlertGlyphicon('critical', _("UCP Node is not running")));
		}

		return array($service);
	}

	/**
	 * Start FreePBX for fwconsole hook
	 * @param object $output The output object.
	 */
	public function startFreepbx($output=null) {
		if(!$this->freepbx->Config->get("NODEJSENABLED")) {
			return;
		}
		$webroot = $this->freepbx->Config->get("AMPWEBROOT");
		$varlibdir = $this->freepbx->Config->get("ASTVARLIBDIR");
		$astlogdir = $this->freepbx->Config->get("ASTLOGDIR");

		if(!file_exists($this->nodeloc."/node_modules/forever")) {
			if(is_object($output)) {
				$output->writeln("<error>"._("UCP Node is damaged. Please reinstall.")."</error>");
			} else {
				outn(_("UCP Node is damaged. Please reinstall."));
			}
			return false;
		}

		if(!is_executable($this->nodeloc."/node_modules/forever/bin/forever")) {
			@chmod($this->nodeloc."/node_modules/forever/bin/forever",0755);
		}
		if(!is_executable($this->nodeloc."/node_modules/forever/bin/monitor")) {
			@chmod($this->nodeloc."/node_modules/forever/bin/monitor",0755);
		}

		if(!is_executable($this->nodeloc."/node_modules/forever/bin/forever")) {
			if(is_object($output)) {
				$output->writeln("<error>"._("Unable to launch UCP Node monitor")."</error>");
			} else {
				outn(_("Unable to launch UCP Node monitor"));
			}
			return false;
		}

		$data = $this->getServiceStatus();
		if(!empty($data)) {
			$uptime = $this->get_date_diff(time(), (int)round($data['ctime']/1000));
			if(is_object($output)) {
				$output->writeln(sprintf(_("UCP Server has already been running on PID %s for %s"),$data['pid'],$uptime));
			}
			return true;
		}

		$command = $this->generateRunAsAsteriskCommand('npm start');
		$process = new Process($command);
		$process->setTimeout(15);

		// executes after the command finishes
		if(is_object($output)) {
			$output->writeln(_("Starting UCP Server..."));
		}

		$process->run();

		if ($process->isSuccessful()) {
			if(is_object($output)) {
				$progress = new ProgressBar($output, 0);
				$progress->setFormat('[%bar%] %elapsed%');
				$progress->start();
			}
			$i = 0;
			while($i < 100) {
				$data = $this->getServiceStatus();
				if(!empty($data)) {
					if(is_object($output)) {
						$progress->finish();
					}
					break;
				}
				if(is_object($output)) {
					$progress->setProgress($i);
				}
				$i++;
				usleep(100000);
			}
			if(is_object($output)) {
				$output->writeln("");
			}
			if(!empty($data)) {
				if(is_object($output)) {
					$output->writeln(sprintf(_("Started UCP Node Server. PID is %s"),$data['pid']));
				}
				return true;
			} else {
				$logPath = $this->freepbx->Config->get("ASTLOGDIR");
				if(is_object($output)) {
					$output->write("<error>".sprintf(_("The process did not stay running. Please check the logs at %s"),$logPath."/ucp_err.log")."</error>");
				} else {
					outn(sprintf(_("The process did not stay running. Please check the logs at %s"),$logPath."/ucp_err.log"));
				}
			}
		} else {
			if(is_object($output)) {
				$output->writeln("<error>".sprintf(_("Failed: %s"),$process->getErrorOutput())."</error>");
			} else {
				outn(sprintf(_("Failed: %s"),$process->getErrorOutput()));
			}
		}
		return false;
	}

	/**
	 * Stop FreePBX for fwconsole hook
	 * @param object $output The output object.
	 */
	public function stopFreepbx($output=null) {
		$webroot = $this->freepbx->Config->get("AMPWEBROOT");
		$varlibdir = $this->freepbx->Config->get("ASTVARLIBDIR");
		$astlogdir = $this->freepbx->Config->get("ASTLOGDIR");
		if(!file_exists($this->nodeloc."/node_modules/forever")) {
			if(is_object($output)) {
				$output->writeln("<error>"._("UCP Node is damaged. Please reinstall.")."</error>");
			}
			return false;
		}

		if(!is_executable($this->nodeloc."/node_modules/forever")) {
			if(is_object($output)) {
				$output->writeln("<error>"._("Unable to launch UCP Node monitor")."</error>");
			}
			return false;
		}

		$data = $this->getServiceStatus();
		if(empty($data)) {
			if(is_object($output)) {
				$output->writeln("<error>"._("UCP Node is not running")."</error>");
			}
			return false;
		}

		$command = $this->generateRunAsAsteriskCommand('npm stop');
		$process = new Process($command);
		$process->setTimeout(15);

		// executes after the command finishes
		if(is_object($output)) {
			$output->writeln(_("Stopping UCP Server"));
		}

		$process->run();

		$data = $this->getServiceStatus();
		if (empty($data) && $process->isSuccessful()) {
			if(is_object($output)) {
				$output->writeln(_("Stopped UCP Server"));
			}
		} else {
			if(is_object($output)) {
				$output->writeln("<error>".sprintf(_("UCP Server Failed: %s")."</error>",$process->getErrorOutput()));
			}
			return false;
		}
		return true;
	}

	/**
	 * FreePBX chown hooks
	 */
	public function chownFreepbx() {
		$files = array();

		$files[] = array('type' => 'execdir',
			'path' => __DIR__.'/node/node_modules/forever/bin',
			'perms' => 0755);

		$files[] = array('type' => 'rdir',
			'path' => $this->foreverroot,
			'perms' => 0755);

		return $files;
	}

	/**
	 * Get human readable time difference between 2 dates
	 *
	 * Return difference between 2 dates in year, month, hour, minute or second
	 * The $precision caps the number of time units used: for instance if
	 * $time1 - $time2 = 3 days, 4 hours, 12 minutes, 5 seconds
	 * - with precision = 1 : 3 days
	 * - with precision = 2 : 3 days, 4 hours
	 * - with precision = 3 : 3 days, 4 hours, 12 minutes
	 *
	 * From: http://www.if-not-true-then-false.com/2010/php-calculate-real-differences-between-two-dates-or-timestamps/
	 *
	 * @param mixed $time1 a time (string or timestamp)
	 * @param mixed $time2 a time (string or timestamp)
	 * @param integer $precision Optional precision
	 * @return string time difference
	 */
	private function get_date_diff( $time1, $time2, $precision = 2 ) {
		// If not numeric then convert timestamps
		if( !is_int( $time1 ) ) {
			$time1 = strtotime( $time1 );
		}
		if( !is_int( $time2 ) ) {
			$time2 = strtotime( $time2 );
		}
		// If time1 > time2 then swap the 2 values
		if( $time1 > $time2 ) {
			list( $time1, $time2 ) = array( $time2, $time1 );
		}
		// Set up intervals and diffs arrays
		$intervals = array( 'year', 'month', 'day', 'hour', 'minute', 'second' );
		$diffs = array();
		foreach( $intervals as $interval ) {
			// Create temp time from time1 and interval
			$ttime = strtotime( '+1 ' . $interval, $time1 );
			// Set initial values
			$add = 1;
			$looped = 0;
			// Loop until temp time is smaller than time2
			while ( $time2 >= $ttime ) {
				// Create new temp time from time1 and interval
				$add++;
				$ttime = strtotime( "+" . $add . " " . $interval, $time1 );
				$looped++;
			}
			$time1 = strtotime( "+" . $looped . " " . $interval, $time1 );
			$diffs[ $interval ] = $looped;
		}
		$count = 0;
		$times = array();
		foreach( $diffs as $interval => $value ) {
			// Break if we have needed precission
			if( $count >= $precision ) {
				break;
			}
			// Add value and interval if value is bigger than 0
			if( $value > 0 ) {
				if( $value != 1 ){
					$interval .= "s";
				}
				// Add value and interval to times array
				$times[] = $value . " " . $interval;
				$count++;
			}
		}
		// Return string with times
		return implode( ", ", $times );
	}

	private function genAlertGlyphicon($res, $tt = null) {
		$glyphs = array(
			"ok" => "glyphicon-ok text-success",
			"warning" => "glyphicon-warning-sign text-warning",
			"error" => "glyphicon-remove text-danger",
			"unknown" => "glyphicon-question-sign text-info",
			"info" => "glyphicon-info-sign text-info",
			"critical" => "glyphicon-fire text-danger"
		);
		// Are we being asked for an alert we actually know about?
		if (!isset($glyphs[$res])) {
			return array('type' => 'unknown', "tooltip" => sprintf(_("Don't know what %s is").$res), "glyph-class" => $glyphs['unknown']);
		}

		if ($tt === null) {
			// No Tooltip
			return array('type' => $res, "tooltip" => null, "glyph-class" => $glyphs[$res]);
		} else {
			// Generate a tooltip
			$html = '';
			if (is_array($tt)) {
				foreach ($tt as $line) {
					$html .= htmlentities($line, ENT_QUOTES, "UTF-8")."\n";
				}
			} else {
				$html .= htmlentities($tt, ENT_QUOTES, "UTF-8");
			}

			return array('type' => $res, "tooltip" => $html, "glyph-class" => $glyphs[$res]);
		}
		return '';
	}

	private function getServiceStatus() {
		foreach(glob($this->foreverroot."/sock/*.sock") as $file) {
			$sock = @stream_socket_client('unix://'.$file, $errno, $errstr);
			if($sock === false) {
				@unlink($file);
				continue;
			}
			fwrite($sock, '["data", {}]'."\r\n");
			$info = fread($sock, 16384)."\n";
			$children = json_decode($info,true);
			if(!is_array($children)) {
				//strange?
				continue;
			}
			array_shift($children);
			foreach($children as $child) {
				if($child['running']) {
					return $child;
				} else {
					exec("kill -9 ".$child['foreverPid']);
					@unlink($file);
					break;
				}
			}
			fclose($sock);
		}
		return false;
	}

	private function generateRunAsAsteriskCommand($command) {
		$webuser = $this->freepbx->Config->get('AMPASTERISKWEBUSER');
		$webgroup = $this->freepbx->Config->get('AMPASTERISKWEBGROUP');
		$webroot = $this->freepbx->Config->get("AMPWEBROOT");
		$varlibdir = $this->freepbx->Config->get("ASTVARLIBDIR");
		$astlogdir = $this->freepbx->Config->get("ASTLOGDIR");

		$cmds = array(
			'cd '.$this->nodeloc,
			'mkdir -p '.$this->foreverroot,
			'mkdir -p '.$this->nodeloc.'/logs',
			'export HOME="'.$this->getHomeDir().'"',
			'echo "prefix = ~/.node" > ~/.npmrc',
			'export FOREVER_ROOT="'.$this->foreverroot.'"',
			'export ASTLOGDIR="'.$astlogdir.'"',
			'export PATH="$HOME/.node/bin:$PATH"',
			'export NODE_PATH="$HOME/.node/lib/node_modules:$NODE_PATH"',
			'export MANPATH="$HOME/.node/share/man:$MANPATH"'
		);
		$cmds[] = $command;
		$final = implode(" && ", $cmds);

		if (posix_getuid() == 0) {
			$final = "runuser -l asterisk -c '".$final."'";
		}
		return $final;
	}
}
